/*
 * FreeRTOS Kernel V10.2.1
 * Copyright (C) 2019 Amazon.com, Inc. or its affiliates.  All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * http://www.FreeRTOS.org
 * http://aws.amazon.com/freertos
 *
 * 1 tab == 4 spaces!
 */

/******************************************************************************
 * This project provides two demo applications.  A simple blinky style project,
 * and a more comprehensive test and demo application.  The
 * mainCREATE_SIMPLE_BLINKY_DEMO_ONLY setting (defined in this file) is used to
 * select between the two.  The simply blinky demo is implemented and described
 * in main_blinky.c.  The more comprehensive test and demo application is
 * implemented and described in main_full.c.
 *
 * This file implements the code that is not demo specific, including the
 * hardware setup, standard FreeRTOS hook functions, and the ISR hander called
 * by the RTOS after interrupt entry (including nesting) has been taken care of.
 *
 * NOTE on LEDS:
 *
 *     This demo is NOT configured to use the LED built onto the SAMA6D2
 *     XPLained board!
 *
 *     The LED driver PIN_LED definitions have been altered in
 *     board_sama5d2-xplained.h to remap them to GPIOs terminating on pins 30,
 *     32 and 34 of J17. (This change is conditional on the preprocessor
 *     #define "LEDS_ON_J17".) These GPIOs are configured to be "high drive"
 *     push-pull outputs; they can source up to 18mA at 1.8v. Low
 *     forward-voltage LEDs may be connected via 100 ohm resistors to pins
 *     30, 32 and 34 with their cathodes to pin 35/36 (GND).
 *
 */

/*
 * Procedure to download and execute code from RAM
 * -----------------------------------------------
 *
 * 1. Close jumper JP9(BOOT_DIS) and JP2(DEBUG_DIS).
 * 2. Open jumper JP1(EDBG_DIS).
 * 3. Power on the board by USB connection on J14(EDBG_JTAG).
 * 4. Open EDBG Virtual COM Port with setting 57600,8,N,1.
 * 5. Type # on HyperTerminal and get > as a reply.
 * 6. Dont reset the board during debugging. For IAR, set the Debugger to CMSIS
 *    DAP, with the "Disabled(no reset)" option.
 */

/* Scheduler include files. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Standard demo includes. */
#include "partest.h"
#include "TimerDemo.h"
#include "QueueOverwrite.h"
#include "EventGroupsDemo.h"

/* Library includes. */
#include "board_sama5d2-xplained.h"
#include "peripherals/wdt.h"
#include "peripherals/pio.h"
#include "chip.h"

/* Set mainCREATE_SIMPLE_BLINKY_DEMO_ONLY to one to run the simple blinky demo,
or 0 to run the more comprehensive test and demo application. */
#define mainCREATE_SIMPLE_BLINKY_DEMO_ONLY	1

/*-----------------------------------------------------------*/

/*
 * Configure the hardware as necessary to run this demo.
 */
static void prvSetupHardware( void );

/*
 * main_blinky() is used when mainCREATE_SIMPLE_BLINKY_DEMO_ONLY is set to 1.
 * main_full() is used when mainCREATE_SIMPLE_BLINKY_DEMO_ONLY is set to 0.
 */
#if mainCREATE_SIMPLE_BLINKY_DEMO_ONLY == 1
	extern void main_blinky( void );
#else
	extern void main_full( void );
#endif /* #if mainCREATE_SIMPLE_BLINKY_DEMO_ONLY == 1 */

/* Prototypes for the standard FreeRTOS callback/hook functions implemented
within this file. */
void vApplicationMallocFailedHook( void );
void vApplicationIdleHook( void );
void vApplicationStackOverflowHook( TaskHandle_t pxTask, char *pcTaskName );
void vApplicationTickHook( void );

/* Prototype for the IRQ handler called by the generic Cortex-A5 RTOS port
layer. */
void vApplicationIRQHandler( void );

/*-----------------------------------------------------------*/

int main( void )
{
	/* Configure the hardware ready to run the demo. */
	prvSetupHardware();

	/* The mainCREATE_SIMPLE_BLINKY_DEMO_ONLY setting is described at the top
	of this file. */
	#if( mainCREATE_SIMPLE_BLINKY_DEMO_ONLY == 1 )
	{
		main_blinky();
	}
	#else
	{
		main_full();
	}
	#endif

	return 0;
}
/*-----------------------------------------------------------*/

static void prvSetupHardware( void )
{
	/* Disable watchdog */
	wdt_disable( );

	/* Set protect mode in the AIC for easier debugging. */
	AIC->AIC_DCR |= AIC_DCR_PROT;

	/* Configure ports used by LEDs. */
	vParTestInitialise();

	#if defined (ddram)
		MMU_Initialize( ( uint32_t * ) 0x30C000 );
		CP15_EnableMMU();
		CP15_EnableDcache();
		CP15_EnableIcache();
	#endif
}
/*-----------------------------------------------------------*/

void vApplicationMallocFailedHook( void )
{
	/* Called if a call to pvPortMalloc() fails because there is insufficient
	free memory available in the FreeRTOS heap.  pvPortMalloc() is called
	internally by FreeRTOS API functions that create tasks, queues, software
	timers, and semaphores.  The size of the FreeRTOS heap is set by the
	configTOTAL_HEAP_SIZE configuration constant in FreeRTOSConfig.h. */

	/* Force an assert. */
	configASSERT( ( volatile void * ) NULL );
}
/*-----------------------------------------------------------*/

void vApplicationStackOverflowHook( TaskHandle_t pxTask, char *pcTaskName )
{
	( void ) pcTaskName;
	( void ) pxTask;

	/* Run time stack overflow checking is performed if
	configCHECK_FOR_STACK_OVERFLOW is defined to 1 or 2.  This hook
	function is called if a stack overflow is detected. */

	/* Force an assert. */
	configASSERT( ( volatile void * ) NULL );
}
/*-----------------------------------------------------------*/

void vApplicationIdleHook( void )
{
volatile size_t xFreeHeapSpace;

	/* This is just a trivial example of an idle hook.  It is called on each
	cycle of the idle task.  It must *NOT* attempt to block.  In this case the
	idle task just queries the amount of FreeRTOS heap that remains.  See the
	memory management section on the http://www.FreeRTOS.org web site for memory
	management options.  If there is a lot of heap memory free then the
	configTOTAL_HEAP_SIZE value in FreeRTOSConfig.h can be reduced to free up
	RAM. */
	xFreeHeapSpace = xPortGetFreeHeapSize();

	/* Remove compiler warning about xFreeHeapSpace being set but never used. */
	( void ) xFreeHeapSpace;
}
/*-----------------------------------------------------------*/

void vAssertCalled( const char * pcFile, unsigned long ulLine )
{
volatile unsigned long ul = 0;

	( void ) pcFile;
	( void ) ulLine;

	taskENTER_CRITICAL();
	{
		/* Set ul to a non-zero value using the debugger to step out of this
		function. */
		while( ul == 0 )
		{
			portNOP();
		}
	}
	taskEXIT_CRITICAL();
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
	#if mainCREATE_SIMPLE_BLINKY_DEMO_ONLY == 0
	{
		/* The full demo includes a software timer demo/test that requires
		prodding periodically from the tick interrupt. */
		vTimerPeriodicISRTests();

		/* Call the periodic queue overwrite from ISR demo. */
		vQueueOverwritePeriodicISRDemo();

		/* Call the periodic event group from ISR demo. */
		vPeriodicEventGroupsProcessing();
	}
	#endif
}
/*-----------------------------------------------------------*/

/* The function called by the RTOS port layer after it has managed interrupt
entry. */
void vApplicationIRQHandler( void )
{
typedef void (*ISRFunction_t)( void );
ISRFunction_t pxISRFunction;
volatile uint32_t * pulAIC_IVR = ( uint32_t * ) configINTERRUPT_VECTOR_ADDRESS;

	/* Obtain the address of the interrupt handler from the AIR. */
	pxISRFunction = ( ISRFunction_t ) *pulAIC_IVR;

	/* Write back to the SAMA5's interrupt controller's IVR register in case the
	CPU is in protect mode.  If the interrupt controller is not in protect mode
	then this write is not necessary. */
	*pulAIC_IVR = ( uint32_t ) pxISRFunction;

	/* Ensure the write takes before re-enabling interrupts. */
	__DSB();
	__ISB();
	__enable_irq();

	/* Call the installed ISR. */
	pxISRFunction();
}

/* Keep the linker quiet. */
size_t __write(int, const unsigned char *, size_t);
size_t __write(int f, const unsigned char *p, size_t s)
{
  (void) f;
  (void) p;
  (void) s;
  return 0;
}



