/*
 * FreeRTOS Kernel V10.2.1
 * Copyright (C) 2019 Amazon.com, Inc. or its affiliates.  All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * http://www.FreeRTOS.org
 * http://aws.amazon.com/freertos
 *
 * 1 tab == 4 spaces!
 */

/*
Changes from V1.01:

	+ Types used updated.
	+ Add vParTestToggleLED();

Changes from V2.0.0

	+ Use scheduler suspends in place of critical sections.
*/


#include "FreeRTOS.h"
#include "task.h"
#include "partest.h"

#define partstALL_OUTPUTS_OFF			( ( unsigned short) 0x00 )
#define partstMAX_OUTPUT_LED			( ( unsigned char ) 7 )
#define partstPORT_F_ADDR				( ( unsigned short ) 0x605 )
#define partstPORT_DIRECTION_REG		( ( unsigned short ) 0x606 )
#define partstPORT_F_DIR_BIT			( ( unsigned short ) 0x20 )

/*lint -e956 File scope parameters okay here. */
static volatile unsigned char ucCurrentOutputValue = partstALL_OUTPUTS_OFF;
/*lint +e956 */

/*-----------------------------------------------------------
 * Simple parallel port IO routines.
 *-----------------------------------------------------------*/

void vParTestInitialise( void )
{
unsigned short usInput;

	ucCurrentOutputValue = partstALL_OUTPUTS_OFF;

	/* Set the direction to output for port F. */
	usInput = portINPUT_BYTE( partstPORT_DIRECTION_REG );
	usInput |= partstPORT_F_DIR_BIT;
	portOUTPUT_BYTE( partstPORT_DIRECTION_REG, usInput );

	/* Start with all outputs off. */
	portOUTPUT_BYTE( partstPORT_F_ADDR, partstALL_OUTPUTS_OFF );
}
/*-----------------------------------------------------------*/

void vParTestSetLED( unsigned portBASE_TYPE uxLED, portBASE_TYPE xValue )
{
unsigned char ucBit = ( unsigned char ) 1;

	if( uxLED <= partstMAX_OUTPUT_LED )
	{
		ucBit <<= uxLED;
	}	

	vTaskSuspendAll();
	{
		if( xValue == pdTRUE )
		{
			ucBit ^= ( unsigned char ) 0xff;
			ucCurrentOutputValue &= ucBit;
		}
		else
		{
			ucCurrentOutputValue |= ucBit;
		}

		portOUTPUT_BYTE( partstPORT_F_ADDR, ( unsigned ) ucCurrentOutputValue );
	}
	xTaskResumeAll();
}
/*-----------------------------------------------------------*/

void vParTestToggleLED( unsigned portBASE_TYPE uxLED )
{
unsigned char ucBit;

	if( uxLED <= partstMAX_OUTPUT_LED )
	{
		ucBit = ( ( unsigned char ) 1 ) << uxLED;

		vTaskSuspendAll();
		{
			if( ucCurrentOutputValue & ucBit )
			{
				ucCurrentOutputValue &= ~ucBit;
			}
			else
			{
				ucCurrentOutputValue |= ucBit;
			}

			portOUTPUT_BYTE( partstPORT_F_ADDR, ( unsigned ) ucCurrentOutputValue );
		}
		xTaskResumeAll();			
	}
}

